# AGENT.md — Spécification d’implémentation pour GitHub Copilot

> Ce document prescrit, étape par étape, ce que Copilot doit générer pour livrer une API Flask de 
> **résumé de textes en français (10–15 mots)** avec **deux modes** (baseline FP32 et optimisé), 
> **mesures par requête** (énergie Wh, latence ms, mémoire MiB si dispo) et 
> **application web intégrée**, conformément au sujet « Design4Green 2025 ». 
> Les exigences normatives citées ci‑dessous proviennent du PDF du sujet (PDF `docs/VF_Sujet Design4Green 2025.pdf`) 

---

## 1) Mandat et périmètre

- Construire une **API Flask** avec un **unique endpoint** `POST /summarize` qui reçoit
  ```json
  {"text": "<=4000 caractères>", "optimized": true|false}
  ```
  et renvoie un résumé **en français** de **10 à 15 mots**, plus les métriques de performance par requête.
- Deux modes à **endpoint unique** :
  - `optimized=false` → **baseline**: **FP32 strict**, aucune optimisation activée.
  - `optimized=true` → **optimisé**: activer des optimisations **CPU** mesurées (p. ex. quantification dynamique INT8, low‑rank/adapters, pruning léger, `torch.compile` si bénéfice).
- **Mesure**: énergie (Wh) via **CodeCarbon** ou équivalent, **latence** (ms), **mémoire** (MiB si disponible), mesurées du **juste avant l’inférence** au **juste après**.
- **Modèle par défaut (obligatoire)**: `EleutherAI/pythia-70m-deduped` exécuté **sur CPU**.
- **Application web intégrée** permettant de tester l’API et d’afficher toutes les métriques.
- **Évaluation**: le jury exécute `python judge.py` contre l’API à `http://127.0.0.1:5000` et contrôle deux sorties: **Score (/100)** et **Économie d’énergie (%)**. **Éligibilité**: au moins **95 %** des résumés doivent contenir **10–15 mots**.

---

## 2) Contrat d’API (à implémenter strictement)cmd /c "cd /d C:\Users\guilh\OneDrive\Dokumente\GitHub\Design2x2Green && pytest tests/test_length_gate.py -q -s"   

### Requête
- Méthode: `POST`
- URL: `/summarize`
- Corps JSON:
  ```json
  {
    "text": "string, <= 4000 caractères, UTF-8",
    "optimized": false
  }
  ```

### Réponse (200 OK)
```json
{
  "summary": "string, français, 10-15 mots",
  "metrics": {
    "energy_Wh": 0.0,
    "latency_ms": 0.0,
    "memory_MiB": 0.0
  },
  "mode": "baseline" | "optimized"
}
```

### Règles de validation
- Rejeter `text` manquant, vide, ou > 4000 caractères avec 400 Bad Request.
- Imposer le français et 10–15 mots **en post‑traitement**. Ne jamais préfixer par « Résumé : ».
- Limiter les répétitions dans le résumé.

---

## 3) Architecture et arborescence du dépôt

Créer exactement l’arborescence suivante:

```
.
├── app.py                        # API Flask (endpoint /summarize)
├── requirements.txt
├── README.md
├── AGENT.md
├── judge.py                      # fourni par l'orga, à ne pas modifier
├── docs                          # répertoire documents de références
│   └── VF_Sujet Design4Green 2025.pdf
├── scripts/
│   ├── start_api.sh              # lance API sur 127.0.0.1:5000
│   ├── start_web.sh              # lance l'application web intégrée
│   └── bench_local.sh            # micro-benchmarks locaux
├── src/
│   ├── __init__.py
│   ├── config.py
│   ├── generation.py             # chargement modèle, modes baseline/optimisé
│   ├── summarizer.py             # pipeline de résumé (pré/post-traitements)
│   └── metrics.py                # instrumentation (énergie, latence, mémoire)
├── web/
│   ├── templates/
│   │   └── index.html
│   ├── static/
│   │   ├── styles.css
│   │   └── app.js
│   └── run.py                    # serveur web minimal (peut réutiliser Flask)
└── tests/
    ├── test_api.py
    └── test_length_gate.py
```

**requirements.txt** minimal et stable :
```
flask
transformers
torch
codecarbon
psutil
numpy
```

---

## 4) Détails d’implémentation

### 4.1 `src/config.py`
- Constantes:
  - `MODEL_NAME = "EleutherAI/pythia-70m-deduped"`
  - `MAX_INPUT_TOKENS = 512`
  - `MIN_WORDS = 10`, `MAX_WORDS = 15`, `TARGET_WORDS = 12`
  - `SEED = 42`
- Environnement reproductible:
  - `PYTHONHASHSEED=0`, graine `torch.manual_seed(SEED)` et `numpy.random.seed(SEED)`
- Paramètres CPU:
  - `OMP_NUM_THREADS = min(4, os.cpu_count())`, fixer aussi `torch.set_num_threads(...)`.

### 4.2 `src/generation.py`
- `load_tokenizer_model()` → `(tokenizer, model)` en **FP32** sur CPU.
- `get_model(mode)` :
  - **baseline** → modèle FP32 **sans** optimisation.
  - **optimized** → appliquer **quantization dynamique INT8** (`torch.ao.quantization.quantize_dynamic` sur `nn.Linear`), puis **optionnellement** `torch.compile(model, mode="reduce-overhead")` si un test local montre un gain stable. Conserver un **cache** des deux graphes (baseline, optimisé) pour éviter re‑quantification.
- Paramétrer `model.eval()`. Ne pas activer de sampling stochastique.

### 4.3 `src/summarizer.py`
- Prétraitement : normalisation Unicode, espaces, ponctuation. Tronquer l’entrée à `MAX_INPUT_TOKENS` via le tokenizer.
- Génération : greedy (`do_sample=False`, `num_beams=1`, `max_new_tokens=64`, `eos_token_id` défini).
- Post‑traitement longueur :
  - Compter les **mots** avec une regex Unicode.
  - Retirer toute signature « Résumé: », capitaliser initiale, ajouter point final si absent.
- Invariance : même entrée + même mode ⇒ même sortie tant que le processus reste chargé.

### 4.4 `src/metrics.py`
- Contexte `tracked_inference()`:
  - Démarrer **CodeCarbon** juste avant la génération, l’arrêter juste après.
  - `energy_Wh` : récupérer l’énergie en kWh et convertir en Wh si nécessaire.
  - `latency_ms` : `time.perf_counter()` autour de la génération.
  - `memory_MiB` : `resource.getrusage` ou `psutil.Process().memory_info().rss/2**20`.
  - Export **append‑only** vers `metrics/history.jsonl` (une ligne JSON par requête).

### 4.5 `app.py`
- Implémenter le **seul** endpoint `POST /summarize`.
- Valider l’entrée, router vers `summarizer.summarize(text, optimized)` enveloppé par `metrics.measure()`.
- Répondre au format JSON prescrit plus haut.

### 4.6 Application web (`web/`)
- `run.py` : réutiliser Flask pour servir `index.html` et proxy vers `/summarize`.
- `index.html` : champ texte, toggle Baseline/Optimisé, bouton « Résumer », zone **Résumé** + **Métriques** (Wh, ms, MiB).
- `styles.css` : design sobre, pas de framework lourd, pas de polices externes.
- `app.js` : `fetch('/summarize', ...)`, affichage des résultats, spinner pendant l’inférence. fileciteturn0file0

---

## 5) Scripts d’automatisation

- `scripts/start_api.sh`
  ```bash
  #!/usr/bin/env bash
  export PYTHONHASHSEED=0
  export OMP_NUM_THREADS=${OMP_NUM_THREADS:-4}
  export FLASK_APP=app.py
  flask run --host=127.0.0.1 --port=5000
  ```

- `scripts/start_web.sh`
  ```bash
  #!/usr/bin/env bash
  export PYTHONHASHSEED=0
  python web/run.py
  ```

- `scripts/bench_local.sh`
  ```bash
  #!/usr/bin/env bash
  python - << 'PY'
  # Envoie N requêtes baseline puis optimisé et agrège moyennes (Wh, ms, MiB).
  PY
  ```

---

## 6) Tests et critères d’acceptation

- `tests/test_api.py` : démarre l’API en local et vérifie :
  - 200 OK, présence des clés `summary` et `metrics`.
  - `metrics.energy_Wh` et `metrics.latency_ms` sont numériques.
  - Comportement identique pour 5 entrées en `optimized=false` puis `true` (métriques ≠ mais contrat identique).

- `tests/test_length_gate.py` :
  - Sur un lot ≥ 40 entrées variées, **≥ 95 %** des sorties contiennent **10–15 mots**. Sinon, échec.
  - le test couvre :
    - Textes de longueur variée (50 à 4000 caractères).
    - Thèmes divers (science, littérature, actualités).
    - le test doit se comporter commme judge.py
  - Cas limites: textes courts (< 10 mots), très longs, ponctuation, accents. 

---

## 7) Procédure d’évaluation externe (référence)

- L’API **doit** écouter sur `http://127.0.0.1:5000`. Le jury lance :
  ```bash
  python judge.py
  ```
  Le script appelle `/summarize` en **baseline** puis en **optimisé** et n’affiche que :
  1) **Score final (/100)**, 2) **Économie d’énergie (%)** vs baseline.
- Conditions d’éligibilité :
  - Langue: français. Longueur: 10–15 mots. Entrée ≤ 4000 caractères.
  - Fenêtre de mesure: du juste avant l’inférence au juste après.
  - **Baseline = FP32 strict**. **Optimisé** = mêmes fonctions, mais avec optimisations CPU mesurées.
  - Reproductibilité: `PYTHONHASHSEED=0` et graine `SEED=42` si sampling. 

---

## 8) Définition de Terminé (DoD)

- API conforme au contrat ci‑dessus. Endpoint unique opérationnel.
- **≥ 95 %** des résumés 10–15 mots sur la suite de tests.
- Web‑app fonctionnelle, sobre, affiche Wh, ms, MiB en temps réel.
- Rapport `report.md` ≥ 2 pages : techniques d’optimisation, hyperparamètres, mesures (Wh, ms, MiB), arbitrages et limites.
- `judge.py` s’exécute **sans modification** et produit un score et une économie d’énergie positives.
- Dépendances gelées dans `requirements.txt`. `README.md` documente lancement API et web. fileciteturn0file0

---

## 9) Pseudocode de référence

```python
# app.py (schéma)
from flask import Flask, request, jsonify
from src.generation import get_model
from src.summarizer import summarize
from src.metrics import measure

app = Flask(__name__)

@app.post("/summarize")
def route():
    data = request.get_json(force=True, silent=False)
    text = data.get("text", "")
    optimized = bool(data.get("optimized", False))
    if not text or len(text) > 4000:
        return jsonify(error="Invalid input"), 400
    mode = "optimized" if optimized else "baseline"
    model, tok = get_model(mode)
    def infer():
        return summarize(model, tok, text, optimized=optimized)
    summary, m = measure(infer)
    return jsonify(summary=summary, metrics=m, mode=mode)
```

---

## 10) Non‑objectifs et garde‑fous

- Pas de GPU, pas de dépendance à des services externes.
- Ne jamais altérer la **baseline FP32**.
- Optimisations **mesurées** seulement: si un « gain » n’est pas observé de manière stable, le laisser désactivé.
- Respect strict du français en sortie et de la fenêtre de mesure. fileciteturn0file0

---

## 11) Aide‑mémoire pour Copilot (prompts suggérés)

- « Crée `app.py` avec un endpoint POST /summarize respectant ce contrat JSON et utilisant src/generation.py, src/summarizer.py, src/metrics.py. »
- « Implémente `generation.py` avec FP32 baseline et une voie quantization dynamique INT8 pour le mode optimisé, avec cache d’instances. »
- « Écris `summarizer.py`: prétraitement, prompt, génération greedy, post‑traitement 10–15 mots, complétion par mots clés si <10. »
- « Instrumente `metrics.py` avec CodeCarbon, perf_counter et psutil/resource. Export JSONL. »
- « Génère `web/index.html`, `web/app.js`, `web/styles.css` pour l’UI minimale, sans framework. »
- « Ajoute `tests/` avec tests d’intégration et de conformité 10–15 mots. »

```

> **Rappel** : en cas de conflit d’interprétation, **se conformer** au PDF `docs/VF_Sujet Design4Green 2025.pdf` et aux citations ci‑dessus.

