# Design4Green 2025 - Résumé de Texte Éco-Responsable

API Flask de résumé de textes en français avec mesure d'énergie et deux modes d'optimisation (baseline FP32 et optimisé INT8).

## 📋 Description

Ce projet implémente une API de résumé de texte qui :
- Génère des résumés en français de 10-15 mots
- Mesure l'énergie (Wh), la latence (ms) et la mémoire (MiB) pour chaque requête
- Propose deux modes : baseline (FP32) et optimisé (INT8 quantization)
- Utilise le modèle `EleutherAI/pythia-70m-deduped` sur CPU
- A noter que ce modèle est imposé par le sujet alors qu'il a été initialement entraîné pour de l'anglais, il ne nous a pas paru permis de réentraîner le modèle sur du français qui n'aurait plus été le même par nature
- Ce qui entraine des résumés idiots dans la majorité des cas, le modèle devant faire avec le peu de mot français qu'il a pu voir durant son entraînement
- Inclut une application web pour tester l'API

## 🚀 Installation

### Prérequis

- Python 3.8+
- pip
- Git LFS (optionnel, voir ci-dessous)

### Installation des dépendances

```bash
pip install -r requirements.txt
```

Les dépendances principales sont :
- flask : API REST
- transformers : Modèles de langage
- torch : Framework de deep learning
- codecarbon : Mesure de l'énergie
- psutil : Mesure de la mémoire
- numpy : Calculs numériques

### Téléchargement du modèle

Le modèle `EleutherAI/pythia-70m-deduped` est nécessaire pour exécuter l'application.

**Option 1 : Téléchargement automatique (recommandé)**

Le modèle sera téléchargé automatiquement depuis HuggingFace lors du premier lancement de l'API si les fichiers locaux ne sont pas disponibles. Aucune action supplémentaire n'est nécessaire.

**Option 2 : Utilisation du modèle inclus via Git LFS**

Si vous avez Git LFS installé, les fichiers du modèle seront téléchargés automatiquement lors du clone du dépôt.

Pour vérifier si Git LFS est installé :
```bash
git lfs version
```

Si Git LFS n'est pas installé, vous pouvez l'installer puis télécharger les fichiers :
```bash
# Installer Git LFS (exemple pour Ubuntu/Debian)
sudo apt-get install git-lfs

# Initialiser Git LFS
git lfs install

# Récupérer les fichiers LFS
git lfs pull
```

**Option 3 : Téléchargement manuel**

Vous pouvez télécharger manuellement le modèle avec le script fourni :
```bash
python download_pythia_model.py
```

## 📖 Utilisation

### Lancement de l'API

**Linux/Mac :**
```bash
./scripts/start_api.sh
```

**Windows :**
```batch
scripts\start_api.bat
```

L'API sera accessible sur `http://127.0.0.1:5000`

Ou manuellement :

**Linux/Mac :**
```bash
export PYTHONHASHSEED=0
export OMP_NUM_THREADS=4
export FLASK_APP=app.py
flask run --host=127.0.0.1 --port=5000
```

**Windows :**
```batch
set PYTHONHASHSEED=0
set OMP_NUM_THREADS=4
set FLASK_APP=app.py
flask run --host=127.0.0.1 --port=5000
```

### Lancement de l'application web

Dans un nouveau terminal :

**Linux/Mac :**
```bash
./scripts/start_web.sh
```

**Windows :**
```batch
scripts\start_web.bat
```

L'application web sera accessible sur `http://127.0.0.1:8000`

### Utilisation de l'API

#### Endpoint `/summarize`

**Requête POST :**

```json
{
  "text": "Votre texte à résumer (max 4000 caractères)",
  "optimized": false
}
```

**Réponse :**

```json
{
  "summary": "Résumé en français de 10-15 mots.",
  "metrics": {
    "energy_Wh": 0.001234,
    "latency_ms": 1234.56,
    "memory_MiB": 12.34
  },
  "mode": "baseline"
}
```

#### Exemple avec curl

```bash
curl -X POST http://127.0.0.1:5000/summarize \
  -H "Content-Type: application/json" \
  -d '{
    "text": "L'intelligence artificielle transforme notre société en automatisant de nombreuses tâches.",
    "optimized": false
  }'
```

#### Exemple avec Python

```python
import requests

response = requests.post(
    "http://127.0.0.1:5000/summarize",
    json={
        "text": "Votre texte ici...",
        "optimized": True  # ou False pour baseline
    }
)

data = response.json()
print(f"Résumé: {data['summary']}")
print(f"Énergie: {data['metrics']['energy_Wh']} Wh")
print(f"Latence: {data['metrics']['latency_ms']} ms")
```

## 🧪 Tests

### Exécution de tous les tests

```bash
pytest tests/ -v
```

### Tests d'API (test_api.py)

Tests d'intégration de l'API :

```bash
pytest tests/test_api.py -v
```

Ces tests vérifient :
- Le bon fonctionnement de l'endpoint `/summarize`
- La validation des entrées
- Les modes baseline et optimisé
- La structure des réponses JSON
- Les métriques de performance

### Tests de conformité (test_length_gate.py)

Tests simulant le comportement de `judge.py` :

```bash
pytest tests/test_length_gate.py -q -s
```

Ces tests vérifient que :
- ≥95% des résumés contiennent 10-15 mots (critère d'éligibilité)
- Les deux modes (baseline et optimisé) respectent ce critère
- Comparaison des performances entre les deux modes

Le test couvre plus de 40 textes variés :
- Longueurs diverses (50 à 4000 caractères)
- Thèmes variés (science, littérature, technologie, actualités, etc.)
- Cas limites (textes courts, très longs, caractères spéciaux)

### Benchmark local

Pour comparer les performances des deux modes :

**Linux/Mac :**
```bash
./scripts/bench_local.sh
```

**Windows :**
```batch
scripts\bench_local.bat
```

Affiche les moyennes d'énergie, latence et mémoire pour chaque mode.

## 📁 Structure du projet

```
.
├── app.py                    # API Flask principale
├── requirements.txt          # Dépendances Python
├── README.md                 # Documentation
├── AGENT.md                  # Spécifications d'implémentation
├── docs/                     # Documents de référence
│   └── VF_Sujet Design4Green 2025.pdf
├── scripts/                  # Scripts d'automatisation
│   ├── start_api.sh          # Lancement de l'API (Linux/Mac)
│   ├── start_api.bat         # Lancement de l'API (Windows)
│   ├── start_web.sh          # Lancement de l'app web (Linux/Mac)
│   ├── start_web.bat         # Lancement de l'app web (Windows)
│   ├── bench_local.sh        # Benchmarks locaux (Linux/Mac)
│   ├── bench_local.bat       # Benchmarks locaux (Windows)
│   └── bench_local.py        # Script Python pour benchmarks
├── src/                      # Code source
│   ├── __init__.py
│   ├── config.py             # Configuration et constantes
│   ├── generation.py         # Chargement du modèle (baseline/optimisé)
│   ├── summarizer.py         # Pipeline de résumé
│   └── metrics.py            # Instrumentation (énergie, latence, mémoire)
├── web/                      # Application web
│   ├── run.py                # Serveur web
│   ├── templates/
│   │   └── index.html        # Interface utilisateur
│   └── static/
│       ├── styles.css        # Styles CSS
│       └── app.js            # JavaScript
└── tests/                    # Tests
    ├── test_api.py           # Tests d'intégration API
    └── test_length_gate.py   # Tests de conformité (simulation judge.py)
```

## ⚙️ Modes de fonctionnement

### Mode Baseline

- Modèle en FP32 strict
- Aucune optimisation
- Référence pour les mesures

### Mode Optimisé

- Quantification dynamique INT8 (via `torch.quantization.quantize_dynamic`)
- Optionnellement `torch.compile` (PyTorch 2.0+)
- Optimisations CPU mesurées

## 📊 Métriques mesurées

- **Énergie (Wh)** : Consommation électrique via CodeCarbon
- **Latence (ms)** : Temps d'exécution de l'inférence
- **Mémoire (MiB)** : Utilisation de la mémoire RAM

Les métriques sont mesurées du juste avant l'inférence au juste après, et sauvegardées dans `metrics/history.jsonl`.

## 🎯 Critères d'éligibilité

Pour être éligible au concours Design4Green 2025 :

1. **Longueur** : ≥95% des résumés doivent contenir 10-15 mots
2. **Langue** : Résumés en français
3. **Entrée** : Textes ≤4000 caractères
4. **Modes** : Baseline (FP32) et optimisé fonctionnels
5. **Mesures** : Énergie, latence, mémoire correctement instrumentées

## 🔧 Configuration

Les paramètres sont définis dans `src/config.py` :

- `MODEL_NAME = "EleutherAI/pythia-70m-deduped"`
- `MAX_INPUT_TOKENS = 512`
- `MIN_WORDS = 10`
- `MAX_WORDS = 15`
- `TARGET_WORDS = 12`
- `SEED = 42`

Variables d'environnement :
- `PYTHONHASHSEED=0` : Reproductibilité
- `OMP_NUM_THREADS=4` : Threads CPU

## 📝 Notes techniques

### Génération de résumés

Le système génère directement des résumés de 10-15 mots grâce à un prompt optimisé qui guide le modèle. Le post-traitement se limite à :
- Nettoyage de base (retrait de préfixes résiduels)
- Capitalisation de la première lettre
- Ajout de ponctuation finale si nécessaire

La qualité et la longueur des résumés dépendent directement des capacités du modèle.

### Reproductibilité

- Graines aléatoires fixées (`SEED=42`)
- Variables d'environnement contrôlées
- Génération déterministe (greedy decoding, `do_sample=False`)

### Optimisations

La quantification INT8 réduit :
- La taille du modèle (~4x)
- L'utilisation mémoire
- Potentiellement l'énergie consommée

## 🐛 Dépannage

### L'API ne démarre pas

Vérifiez que le port 5000 est disponible :

```bash
lsof -i :5000
```

### Erreur de mémoire

Réduisez `OMP_NUM_THREADS` dans `src/config.py` ou les variables d'environnement.

### Les tests échouent

Assurez-vous que l'API n'est pas déjà en cours d'exécution, car les tests démarrent leur propre instance.

## 📄 Licence

Ce projet est développé dans le cadre du concours Design4Green 2025.

## 🤝 Contribution

Projet académique - Pas de contributions externes.

## 📧 Contact

Pour plus d'informations, consultez `AGENT.md` et le document PDF dans `docs/`.
