"""Flask API for text summarization with energy metrics."""
from flask import Flask, request, jsonify
from src.generation import get_model
from src.summarizer import summarize
from src.metrics import measure
from src.config import setup_environment

# Initialize deterministic environment as early as possible
setup_environment()

app = Flask(__name__)


@app.route("/summarize", methods=["POST"])
def summarize_endpoint():
    """Summarize text endpoint.
    
    Request JSON:
        {
            "text": "string, <= 4000 characters",
            "optimized": false
        }
    
    Response JSON:
        {
            "summary": "string, French, 10-15 words",
            "metrics": {
                "energy_Wh": 0.0,
                "latency_ms": 0.0,
                "memory_MiB": 0.0
            },
            "mode": "baseline" | "optimized"
        }
    """
    # Parse request
    try:
        data = request.get_json(force=True, silent=False)
    except Exception:
        return jsonify({"error": "Invalid JSON"}), 400
    
    # Validate input
    text = data.get("text", "")
    
    if not text:
        return jsonify({"error": "Missing 'text' field"}), 400
    
    if not isinstance(text, str):
        return jsonify({"error": "'text' must be a string"}), 400
    
    if len(text) > 4000:
        return jsonify({"error": "Text exceeds 4000 characters"}), 400
    
    # Get mode
    optimized = bool(data.get("optimized", False))
    mode = "optimized" if optimized else "baseline"
    
    # Get model
    try:
        tokenizer, model = get_model(mode)
    except Exception:
        return jsonify({"error": "Failed to load model"}), 500
    
    # Define inference function
    def infer():
        return summarize(model, tokenizer, text, optimized=optimized)
    
    # Measure and generate summary
    try:
        summary, metrics = measure(infer)
    except Exception:
        return jsonify({"error": "Summarization failed"}), 500
    
    # Return response
    return jsonify({
        "summary": summary,
        "metrics": metrics,
        "mode": mode
    })


@app.route("/health", methods=["GET"])
def health():
    """Health check endpoint."""
    return jsonify({"status": "ok"})


if __name__ == "__main__":
    app.run(host="127.0.0.1", port=5000)