import requests
import json
import time

API_URL = "http://127.0.0.1:5000/summarize"

# Sample texts for testing
test_texts = [
    "L'intelligence artificielle transforme rapidement notre société en automatisant de nombreuses tâches. Cependant, son développement pose des questions éthiques importantes concernant la vie privée, la surveillance et l'emploi.",
    "Le changement climatique est l'un des plus grands défis de notre époque. Les émissions de gaz à effet de serre continuent d'augmenter malgré les avertissements des scientifiques sur les conséquences catastrophiques pour la planète.",
    "Python est un langage de programmation polyvalent utilisé dans de nombreux domaines comme le développement web, l'analyse de données et l'intelligence artificielle. Sa syntaxe simple le rend accessible aux débutants.",
]

def benchmark_mode(mode_name, optimized):
    """Benchmark a specific mode."""
    print(f"\n{'='*60}")
    print(f"Testing {mode_name} mode")
    print('='*60)
    
    metrics_list = []
    
    for i, text in enumerate(test_texts, 1):
        print(f"\nRequest {i}/{len(test_texts)}...")
        
        payload = {
            "text": text,
            "optimized": optimized
        }
        
        try:
            response = requests.post(API_URL, json=payload, timeout=60)
            
            if response.status_code == 200:
                data = response.json()
                print(f"  Summary: {data['summary']}")
                print(f"  Energy: {data['metrics']['energy_Wh']:.6f} Wh")
                print(f"  Latency: {data['metrics']['latency_ms']:.2f} ms")
                print(f"  Memory: {data['metrics']['memory_MiB']:.2f} MiB")
                metrics_list.append(data['metrics'])
            else:
                print(f"  Error: {response.status_code} - {response.text}")
        except Exception as e:
            print(f"  Error: {e}")
    
    # Calculate averages
    if metrics_list:
        avg_energy = sum(m['energy_Wh'] for m in metrics_list) / len(metrics_list)
        avg_latency = sum(m['latency_ms'] for m in metrics_list) / len(metrics_list)
        avg_memory = sum(m['memory_MiB'] for m in metrics_list) / len(metrics_list)
        
        print(f"\n{mode_name} Mode - Average Metrics:")
        print(f"  Energy: {avg_energy:.6f} Wh")
        print(f"  Latency: {avg_latency:.2f} ms")
        print(f"  Memory: {avg_memory:.2f} MiB")
        
        return avg_energy, avg_latency, avg_memory
    
    return None, None, None

# Check if API is available
try:
    response = requests.get("http://127.0.0.1:5000/health", timeout=5)
    if response.status_code != 200:
        print("Error: API is not responding correctly")
        exit(1)
except Exception as e:
    print(f"Error: Cannot connect to API at {API_URL}")
    print("Please start the API first with:")
    print("  Linux/Mac: ./scripts/start_api.sh")
    print("  Windows: scripts/start_api.bat")
    exit(1)

print("="*60)
print("Design4Green Local Benchmark")
print("="*60)
print(f"Testing API at: {API_URL}")
print(f"Number of test texts: {len(test_texts)}")

# Benchmark baseline mode
baseline_energy, baseline_latency, baseline_memory = benchmark_mode("Baseline", False)

# Wait a bit between modes
time.sleep(2)

# Benchmark optimized mode
optimized_energy, optimized_latency, optimized_memory = benchmark_mode("Optimized", True)

# Compare results
if baseline_energy and optimized_energy:
    print(f"\n{'='*60}")
    print("Comparison - Baseline vs Optimized")
    print('='*60)
    
    energy_savings = ((baseline_energy - optimized_energy) / baseline_energy) * 100
    latency_change = ((optimized_latency - baseline_latency) / baseline_latency) * 100
    memory_change = ((optimized_memory - baseline_memory) / baseline_memory) * 100
    
    print(f"Energy savings: {energy_savings:+.2f}%")
    print(f"Latency change: {latency_change:+.2f}%")
    print(f"Memory change: {memory_change:+.2f}%")
    print('='*60)
