"""Configuration module for the Design4Green summarization API."""
import os
import torch
import numpy as np
from pathlib import Path

# Model configuration
MODEL_NAME = "EleutherAI/pythia-70m-deduped"

# Try to use local model path first if it exists
_project_root = Path(__file__).parent.parent
_local_model_path = _project_root / "models" / "cache" / MODEL_NAME
if _local_model_path.exists():
    MODEL_PATH = str(_local_model_path)
else:
    # Fallback to HuggingFace model name
    MODEL_PATH = MODEL_NAME

MAX_INPUT_TOKENS = 512

# Prompt configuration
# Maximum characters to include in prompt (avoids token overflow while preserving context)
MAX_PROMPT_CHARS = 400

# Summary length constraints
MIN_WORDS = 10
MAX_WORDS = 15
TARGET_WORDS = 12

# Reproducibility
SEED = 42

# CPU configuration
OMP_NUM_THREADS = min(4, os.cpu_count() or 4)

def setup_environment():
    """Set up environment for reproducible execution."""
    # Set environment variables
    os.environ["PYTHONHASHSEED"] = "0"
    os.environ["OMP_NUM_THREADS"] = str(OMP_NUM_THREADS)
    
    # Set random seeds
    torch.manual_seed(SEED)
    np.random.seed(SEED)
    
    # Configure PyTorch for CPU
    torch.set_num_threads(OMP_NUM_THREADS)
    
    # Ensure deterministic behavior
    if hasattr(torch, 'use_deterministic_algorithms'):
        try:
            torch.use_deterministic_algorithms(True)
        except Exception:
            # Some operations may not support deterministic mode
            pass
