"""API integration tests for the Design4Green summarization API."""
import pytest
import requests
import time
import subprocess
import os
import signal
from pathlib import Path

# API configuration
API_URL = "http://127.0.0.1:5000"
API_TIMEOUT = 60


class TestAPI:
    """Test suite for the summarization API."""
    
    @pytest.fixture(scope="class")
    def api_server(self):
        """Start the API server for testing."""
        # Change to project directory
        project_dir = Path(__file__).parent.parent
        
        env = os.environ.copy()
        env["PYTHONHASHSEED"] = "0"
        env["FLASK_APP"] = "app.py"
        
        # Start the API server
        process = subprocess.Popen(
            ["python", "-m", "flask", "run", "--host=127.0.0.1", "--port=5000"],
            cwd=project_dir,
            env=env,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        
        # Wait for server to start
        max_retries = 30
        for i in range(max_retries):
            try:
                response = requests.get(f"{API_URL}/health", timeout=1)
                if response.status_code == 200:
                    break
            except Exception:
                pass
            time.sleep(1)
        else:
            process.kill()
            raise RuntimeError("API server failed to start")
        
        yield process
        
        # Cleanup
        process.send_signal(signal.SIGTERM)
        try:
            process.wait(timeout=5)
        except subprocess.TimeoutExpired:
            process.kill()
    
    def test_health_endpoint(self, api_server):
        """Test the health check endpoint."""
        response = requests.get(f"{API_URL}/health", timeout=5)
        assert response.status_code == 200
        data = response.json()
        assert data["status"] == "ok"
    
    def test_summarize_baseline(self, api_server):
        """Test summarization in baseline mode."""
        text = "L'intelligence artificielle transforme notre société en automatisant de nombreuses tâches et processus. Les entreprises utilisent l'IA pour améliorer leur efficacité et créer de nouveaux produits innovants."
        
        response = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 200
        data = response.json()
        
        # Check response structure
        assert "summary" in data
        assert "metrics" in data
        assert "mode" in data
        
        # Check mode
        assert data["mode"] == "baseline"
        
        # Check summary
        summary = data["summary"]
        assert isinstance(summary, str)
        assert len(summary) > 0
        
        # Count words
        word_count = len(summary.split())
        assert 10 <= word_count <= 15, f"Summary has {word_count} words, expected 10-15"
        
        # Check metrics
        metrics = data["metrics"]
        assert "energy_Wh" in metrics
        assert "latency_ms" in metrics
        assert "memory_MiB" in metrics
        
        assert isinstance(metrics["energy_Wh"], (int, float))
        assert isinstance(metrics["latency_ms"], (int, float))
        assert isinstance(metrics["memory_MiB"], (int, float))
        
        assert metrics["latency_ms"] > 0
    
    def test_summarize_optimized(self, api_server):
        """Test summarization in optimized mode."""
        text = "Le réchauffement climatique est causé par les émissions de gaz à effet de serre. Les scientifiques appellent à une action urgente pour limiter l'augmentation de la température mondiale."
        
        response = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": True},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 200
        data = response.json()
        
        # Check mode
        assert data["mode"] == "optimized"
        
        # Check summary
        summary = data["summary"]
        word_count = len(summary.split())
        assert 10 <= word_count <= 15, f"Summary has {word_count} words, expected 10-15"
        
        # Check metrics
        metrics = data["metrics"]
        assert metrics["latency_ms"] > 0
    
    def test_invalid_input_empty_text(self, api_server):
        """Test with empty text."""
        response = requests.post(
            f"{API_URL}/summarize",
            json={"text": "", "optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 400
        data = response.json()
        assert "error" in data
    
    def test_invalid_input_too_long(self, api_server):
        """Test with text exceeding 4000 characters."""
        text = "a" * 4001
        
        response = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 400
        data = response.json()
        assert "error" in data
    
    def test_invalid_input_missing_text(self, api_server):
        """Test with missing text field."""
        response = requests.post(
            f"{API_URL}/summarize",
            json={"optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 400
    
    def test_multiple_requests_same_mode(self, api_server):
        """Test multiple requests in the same mode."""
        texts = [
            "Python est un langage de programmation populaire utilisé dans de nombreux domaines.",
            "La science des données utilise des algorithmes et des statistiques pour analyser les informations.",
            "Les réseaux de neurones artificiels sont inspirés du fonctionnement du cerveau humain.",
        ]
        
        for text in texts:
            response = requests.post(
                f"{API_URL}/summarize",
                json={"text": text, "optimized": False},
                timeout=API_TIMEOUT
            )
            
            assert response.status_code == 200
            data = response.json()
            
            # Check word count
            word_count = len(data["summary"].split())
            assert 10 <= word_count <= 15
            
            # Check metrics exist
            assert data["metrics"]["latency_ms"] > 0
    
    def test_mode_comparison(self, api_server):
        """Test that both modes work with the same text."""
        text = "L'énergie renouvelable provient de sources naturelles comme le soleil et le vent. Elle est essentielle pour réduire notre dépendance aux combustibles fossiles."
        
        # Test baseline
        response_baseline = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response_baseline.status_code == 200
        data_baseline = response_baseline.json()
        
        # Test optimized
        response_optimized = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": True},
            timeout=API_TIMEOUT
        )
        
        assert response_optimized.status_code == 200
        data_optimized = response_optimized.json()
        
        # Both should produce valid summaries
        baseline_words = len(data_baseline["summary"].split())
        optimized_words = len(data_optimized["summary"].split())
        
        assert 10 <= baseline_words <= 15
        assert 10 <= optimized_words <= 15
        
        # Both should have metrics
        assert data_baseline["metrics"]["latency_ms"] > 0
        assert data_optimized["metrics"]["latency_ms"] > 0
    
    def test_special_characters(self, api_server):
        """Test with special characters and accents."""
        text = "L'éducation est essentielle pour le développement économique et social. Les élèves d'aujourd'hui sont les leaders de demain."
        
        response = requests.post(
            f"{API_URL}/summarize",
            json={"text": text, "optimized": False},
            timeout=API_TIMEOUT
        )
        
        assert response.status_code == 200
        data = response.json()
        
        word_count = len(data["summary"].split())
        assert 10 <= word_count <= 15


if __name__ == "__main__":
    pytest.main([__file__, "-v"])
