"""Length gate tests - simulating judge.py evaluation criteria.

This test suite ensures that at least 95% of summaries contain 10-15 words,
which is the main eligibility criterion for the Design4Green competition.
"""
import pytest
import requests
import time
import subprocess
import os
import signal
from pathlib import Path
import datetime
import json

# API configuration
API_URL = "http://127.0.0.1:5000"
API_TIMEOUT = 60

# Test data - diverse texts covering various topics and lengths
TEST_TEXTS = [
    # Science (50-500 chars)
    "La photosynthèse est le processus par lequel les plantes convertissent la lumière solaire en énergie chimique.",
    "Les trous noirs sont des régions de l'espace où la gravité est si forte que même la lumière ne peut s'échapper. Ils se forment généralement lorsqu'une étoile massive s'effondre.",
    "L'ADN contient toutes les informations génétiques nécessaires au développement et au fonctionnement des organismes vivants. Il est composé de quatre bases azotées qui forment des paires spécifiques.",
    
    # Literature (500-1500 chars)
    "Victor Hugo est l'un des plus grands écrivains français du XIXe siècle. Il a écrit des romans emblématiques comme Les Misérables et Notre-Dame de Paris. Son œuvre traite de thèmes universels comme la justice sociale, l'amour et la rédemption. Hugo était également un homme politique engagé qui défendait les droits des pauvres et s'opposait à la peine de mort.",
    "La poésie romantique française du XIXe siècle se caractérise par l'expression des sentiments personnels, la célébration de la nature et la quête de liberté. Les poètes comme Lamartine, Musset et Vigny ont révolutionné la forme poétique traditionnelle en introduisant une plus grande liberté dans le vers et en explorant des thèmes nouveaux. Ils ont privilégié l'émotion sur la raison et ont cherché à exprimer les tourments de l'âme humaine.",
    
    # Technology (200-800 chars)
    "L'informatique quantique promet de révolutionner le calcul en exploitant les propriétés de la mécanique quantique. Les ordinateurs quantiques utilisent des qubits qui peuvent être dans plusieurs états simultanément, contrairement aux bits classiques.",
    "La blockchain est une technologie de stockage et de transmission d'informations, transparente et sécurisée, qui fonctionne sans organe central de contrôle. Elle permet de créer des registres distribués résistants à la modification des données.",
    "Les réseaux de neurones profonds sont des modèles d'apprentissage automatique inspirés du cerveau humain. Ils sont composés de multiples couches de neurones artificiels qui transforment progressivement les données d'entrée pour produire des prédictions.",
    
    # Current events (300-1000 chars)
    "Le changement climatique affecte déjà notre planète de multiples façons. Les températures moyennes augmentent, les glaciers fondent, le niveau des mers s'élève et les phénomènes météorologiques extrêmes deviennent plus fréquents.",
    "La transition énergétique vers les sources d'énergie renouvelables est devenue une priorité mondiale. Les pays investissent massivement dans l'éolien, le solaire et l'hydroélectrique pour réduire leur dépendance aux combustibles fossiles.",
    "L'économie circulaire propose un modèle de production et de consommation qui implique le partage, la location, la réutilisation, la réparation et le recyclage des produits existants aussi longtemps que possible.",
    
    # History (400-1200 chars)
    "La Révolution française de 1789 a profondément transformé la société et le système politique français. Elle a mis fin à la monarchie absolue et proclamé les principes de liberté, d'égalité et de fraternité qui restent les fondements de la République française.",
    "La Renaissance italienne du XVe siècle a été une période de renouveau artistique, culturel et scientifique. Des artistes comme Léonard de Vinci et Michel-Ange ont créé des œuvres qui continuent d'influencer l'art aujourd'hui.",
    
    # Geography (250-700 chars)
    "La forêt amazonienne est le plus grand écosystème de forêt tropicale au monde. Elle abrite une biodiversité exceptionnelle avec des millions d'espèces de plantes, d'animaux et d'insectes. La déforestation menace cet écosystème vital.",
    "Les océans couvrent environ 71% de la surface de la Terre et contiennent 97% de l'eau de notre planète. Ils jouent un rôle crucial dans la régulation du climat et abritent une immense diversité de vie marine.",
    
    # Economy (350-900 chars)
    "L'intelligence artificielle transforme le monde du travail en automatisant de nombreuses tâches répétitives. Cela soulève des questions sur l'avenir de l'emploi et la nécessité de reconvertir les travailleurs vers de nouveaux métiers.",
    "Le commerce électronique a connu une croissance explosive au cours de la dernière décennie. Les plateformes en ligne permettent aux consommateurs d'acheter des produits du monde entier depuis leur domicile.",
    
    # Health (300-1000 chars)
    "Le système immunitaire protège notre corps contre les infections et les maladies. Il comprend différents types de cellules et de protéines qui travaillent ensemble pour identifier et éliminer les agents pathogènes.",
    "L'exercice physique régulier apporte de nombreux bienfaits pour la santé physique et mentale. Il réduit le risque de maladies cardiovasculaires, améliore l'humeur et renforce le système immunitaire.",
    
    # Education (200-600 chars)
    "L'apprentissage en ligne s'est considérablement développé ces dernières années, offrant aux étudiants du monde entier l'accès à des cours et des ressources éducatives de qualité.",
    "L'éducation aux sciences, technologies, ingénierie et mathématiques (STEM) est devenue une priorité dans de nombreux pays pour préparer les jeunes aux emplois de demain.",
    
    # Culture (400-1000 chars)
    "Le cinéma français a une riche histoire qui remonte aux frères Lumière et à l'invention du cinématographe. De la Nouvelle Vague aux films contemporains, le cinéma français a toujours été reconnu pour sa créativité et son audace artistique.",
    "La musique classique européenne a évolué à travers plusieurs périodes distinctes, de la musique baroque de Bach aux compositions romantiques de Beethoven, en passant par l'élégance du classicisme de Mozart.",
    
    # Social issues (350-950 chars)
    "L'égalité des genres reste un défi majeur dans de nombreuses sociétés. Les femmes continuent de faire face à des discriminations dans l'emploi, l'éducation et la représentation politique.",
    "L'urbanisation rapide pose des défis en termes de logement, de transport et d'accès aux services publics. Les villes doivent développer des solutions durables pour accueillir une population croissante.",
    
    # Environment (300-800 chars)
    "La pollution plastique des océans est devenue un problème environnemental majeur. Des millions de tonnes de plastique finissent dans les océans chaque année, menaçant la vie marine et les écosystèmes.",
    "La biodiversité décline à un rythme alarmant en raison de la destruction des habitats, du changement climatique et de la pollution. La protection des espèces menacées est devenue une priorité urgente.",
    
    # Edge cases - very short texts
    "L'eau est essentielle à la vie.",
    "Les ordinateurs sont utiles.",
    "Le soleil brille aujourd'hui dans le ciel bleu.",
    
    # Edge cases - long texts (approaching 4000 chars)
    """La révolution industrielle a commencé au XVIIIe siècle en Grande-Bretagne avant de se propager à travers l'Europe et l'Amérique du Nord. Cette période de transformation profonde a vu l'introduction de nouvelles technologies de production, en particulier dans l'industrie textile et métallurgique. L'invention de la machine à vapeur par James Watt a été un tournant majeur, permettant de mécaniser de nombreux processus auparavant effectués manuellement. Les usines ont commencé à remplacer les ateliers artisanaux, entraînant un déplacement massif de population des campagnes vers les villes. Cette urbanisation rapide a créé de nouveaux défis sociaux et environnementaux. Les conditions de travail dans les premières usines étaient souvent terribles, avec de longues heures de travail, des salaires bas et des environnements dangereux. Le travail des enfants était répandu et accepté. Cependant, ces conditions ont également donné naissance aux premiers mouvements ouvriers et syndicats qui luttaient pour de meilleures conditions de travail et de vie. Sur le plan économique, la révolution industrielle a conduit à une augmentation sans précédent de la productivité et de la richesse. Les nouvelles méthodes de production de masse ont rendu les biens manufacturés plus abordables pour une plus grande partie de la population. Le développement des chemins de fer a révolutionné le transport des marchandises et des personnes, facilitant le commerce et l'expansion des marchés. Les innovations dans les domaines de la communication, comme le télégraphe, ont permis une transmission plus rapide de l'information. La révolution industrielle a également eu des impacts environnementaux significatifs. L'utilisation intensive du charbon a entraîné une pollution atmosphérique dans les centres industriels. La déforestation s'est accélérée pour fournir du bois aux usines et créer des terres agricoles pour nourrir la population urbaine croissante. Les rivières ont été polluées par les déchets industriels. Ces problèmes environnementaux sont devenus de plus en plus visibles et préoccupants au fil du temps. Sur le plan social, la révolution industrielle a transformé la structure de la société. Une nouvelle classe moyenne urbaine a émergé, composée de commerçants, de professionnels et d'entrepreneurs. La classe ouvrière urbaine s'est également développée, vivant souvent dans des conditions précaires dans les quartiers industriels surpeuplés. Les inégalités économiques se sont accentuées, avec une concentration de richesse entre les mains des propriétaires d'usines et des capitalistes. Cette période a également vu des progrès dans l'éducation et la science. Les besoins de l'industrie ont encouragé le développement de l'enseignement technique et scientifique.""",
    
    # Texts with special formatting and punctuation
    "Est-ce que l'intelligence artificielle peut vraiment remplacer le jugement humain? Cette question soulève des débats importants dans de nombreux domaines, de la médecine au droit.",
    "Les trois piliers du développement durable sont: l'économie, l'environnement et le social. Ces dimensions doivent être équilibrées pour assurer un avenir viable pour notre planète.",
    """Le réchauffement climatique - un phénomène observé depuis le milieu du XXe siècle - se caractérise par une augmentation de la température moyenne de l'atmosphère et des océans. Les scientifiques s'accordent sur le fait que les activités humaines, notamment l'émission de gaz à effet de serre, en sont la cause principale.""",
    
    # Texts with numbers and dates
    "En 2024, plus de 5 milliards de personnes utilisent Internet dans le monde. Cette connectivité mondiale a transformé la façon dont nous communiquons, travaillons et accédons à l'information.",
    "Le 14 juillet 1789 marque la prise de la Bastille et le début de la Révolution française. Cet événement historique a eu des répercussions dans toute l'Europe et au-delà.",
    
    # Additional test cases to reach 50 total (14 more cases)
    "Le café est une boisson stimulante consommée dans le monde entier.",
    "Les abeilles jouent un rôle essentiel dans la pollinisation des plantes.",
    "Le chocolat provient des fèves de cacao cultivées dans les régions tropicales.",
    "La Tour Eiffel est le monument le plus visité de Paris.",
    "Les énergies renouvelables incluent le solaire et l'éolien.",
    "La lecture développe l'imagination et enrichit le vocabulaire.",
    "Le recyclage permet de réduire les déchets et préserver l'environnement.",
    "Les mathématiques sont fondamentales pour comprendre le monde.",
    "La démocratie repose sur le vote et la liberté d'expression.",
    "Les vaccins ont sauvé des millions de vies dans le monde.",
    "Internet a révolutionné la communication et l'accès à l'information.",
    "Le sport améliore la santé physique et mentale des individus.",
    "La musique est un langage universel qui transcende les frontières.",
    "Les langues étrangères ouvrent des portes vers d'autres cultures.",
]

# Allow quick runs by limiting sample size via env var TEST_SAMPLE_SIZE
_sample_size = int(os.environ.get("TEST_SAMPLE_SIZE", "0"))
if _sample_size > 0:
    TEST_TEXTS = TEST_TEXTS[:_sample_size]


def now():
    return datetime.datetime.now().isoformat(sep=' ', timespec='seconds')


def log(msg):
    print(f"[{now()}] {msg}")


def pretty_metrics(m):
    try:
        return f"energy={m.get('energy_Wh', 0):.6f}Wh, latency={m.get('latency_ms', 0):.2f}ms, memory={m.get('memory_MiB', 0):.2f}MiB"
    except Exception:
        return str(m)


class TestLengthGate:
    """Test suite to verify that 95% of summaries meet the 10-15 word requirement."""
    
    @pytest.fixture(scope="class")
    def api_server(self):
        """Start the API server for testing."""
        project_dir = Path(__file__).parent.parent
        
        env = os.environ.copy()
        env["PYTHONHASHSEED"] = "0"
        env["FLASK_APP"] = "app.py"
        
        log("Starting Flask API server for tests...")
        process = subprocess.Popen(
            ["python", "-m", "flask", "run", "--host=127.0.0.1", "--port=5000"],
            cwd=project_dir,
            env=env,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        log(f"Started process PID={process.pid}")

        # Wait for server to start
        max_retries = 30
        server_ready = False
        for i in range(max_retries):
            try:
                response = requests.get(f"{API_URL}/health", timeout=1)
                log(f"Health check attempt {i+1}/{max_retries} -> status {getattr(response, 'status_code', 'N/A')}")
                if response.status_code == 200:
                    server_ready = True
                    log("API server is ready")
                    break
            except Exception as e:
                log(f"Health check attempt {i+1} failed: {e}")
            time.sleep(1)
        if not server_ready:
            process.kill()
            raise RuntimeError("API server failed to start")
        
        yield process
        
        # Cleanup
        log("Stopping API server...")
        try:
            process.send_signal(signal.SIGTERM)
            process.wait(timeout=5)
            log("API server stopped gracefully")
        except subprocess.TimeoutExpired:
            process.kill()
            log("API server killed after timeout")

    def count_words(self, text):
        """Count words in text."""
        return len(text.split())
    
    def is_valid_length(self, text):
        """Check if text has 10-15 words."""
        word_count = self.count_words(text)
        return 10 <= word_count <= 15
    
    def _post_and_log(self, text, optimized):
        """Helper to post to /summarize and return (success, data_or_error, duration_s)."""
        payload = {"text": text, "optimized": optimized}
        start = time.perf_counter()
        try:
            resp = requests.post(f"{API_URL}/summarize", json=payload, timeout=API_TIMEOUT)
            duration = time.perf_counter() - start
        except Exception as e:
            duration = time.perf_counter() - start
            return False, f"Request exception: {e}", duration
        if resp.status_code != 200:
            return False, f"Bad status: {resp.status_code}, body={resp.text[:200]}", duration
        try:
            data = resp.json()
        except Exception as e:
            return False, f"JSON decode error: {e}, body={resp.text[:500]}", duration
        return True, data, duration

    def test_length_gate_baseline(self, api_server):
        """Test that ≥95% of summaries in baseline mode have 10-15 words."""
        valid_count = 0
        total_count = len(TEST_TEXTS)
        results = []
        
        log("\n" + "="*70)
        log("Testing Baseline Mode - Length Gate (≥95% must be 10-15 words)")
        log("="*70)

        for i, text in enumerate(TEST_TEXTS, 1):
            log(f"Test {i}/{total_count} - input_len={len(text)} chars")
            success, data_or_err, duration = self._post_and_log(text, optimized=False)
            if not success:
                log(f"✗ Request failed in {duration:.3f}s: {data_or_err}")
                results.append({"summary": "", "word_count": 0, "valid": False, "error": data_or_err})
                continue
            data = data_or_err
            summary = data.get("summary", "")
            word_count = self.count_words(summary)
            is_valid = self.is_valid_length(summary)
            metrics = data.get("metrics", {})
            results.append({"summary": summary, "word_count": word_count, "valid": is_valid, "metrics": metrics})
            if is_valid:
                valid_count += 1
            status = "✓" if is_valid else "✗"
            log(f"{status} Received in {duration*1000:.1f} ms - words={word_count} - metrics: {pretty_metrics(metrics)}")
            log(f"Summary full: {summary}")
            # Live pass rate
            live_rate = (valid_count / i) * 100
            log(f"Live pass rate after {i} tests: {live_rate:.2f}% ({valid_count}/{i})")

        # Calculate pass rate
        pass_rate = (valid_count / total_count) * 100
        
        log("\n" + "="*70)
        log("Baseline Mode Results")
        log("="*70)
        log(f"Total tests: {total_count}")
        log(f"Valid summaries (10-15 words): {valid_count}")
        log(f"Pass rate: {pass_rate:.2f}%")
        log("Required: ≥95%")
        log("="*70)

        # Show failed cases
        failed = [r for r in results if not r["valid"] and r.get("summary")]
        if failed:
            log(f"\nFailed summaries ({len(failed)}):")
            for i, r in enumerate(failed, 1):
                log(f"{i}. ({r['word_count']} words) {r['summary']} - metrics: {pretty_metrics(r.get('metrics', {}))}")

        # Show requests that errored (no summary)
        errors = [r for r in results if not r["summary"] and not r["valid"]]
        if errors:
            log(f"\nRequests that failed or returned empty summary: {len(errors)}")
            for i, r in enumerate(errors, 1):
                log(f"{i}. error: {r.get('error', 'unknown')}")

        # Assert ≥95% pass rate
        assert pass_rate >= 95.0, f"Pass rate {pass_rate:.2f}% < 95%"
    
    def test_length_gate_optimized(self, api_server):
        """Test that ≥95% of summaries in optimized mode have 10-15 words."""
        valid_count = 0
        total_count = len(TEST_TEXTS)
        results = []
        
        log("\n" + "="*70)
        log("Testing Optimized Mode - Length Gate (≥95% must be 10-15 words)")
        log("="*70)

        for i, text in enumerate(TEST_TEXTS, 1):
            log(f"Test {i}/{total_count} - input_len={len(text)} chars")
            success, data_or_err, duration = self._post_and_log(text, optimized=True)
            if not success:
                log(f"✗ Request failed in {duration:.3f}s: {data_or_err}")
                results.append({"summary": "", "word_count": 0, "valid": False, "error": data_or_err})
                continue
            data = data_or_err
            summary = data.get("summary", "")
            word_count = self.count_words(summary)
            is_valid = self.is_valid_length(summary)
            metrics = data.get("metrics", {})
            results.append({"summary": summary, "word_count": word_count, "valid": is_valid, "metrics": metrics})
            if is_valid:
                valid_count += 1
            status = "✓" if is_valid else "✗"
            log(f"{status} Received in {duration*1000:.1f} ms - words={word_count} - metrics: {pretty_metrics(metrics)}")
            log(f"Summary full: {summary}")
            live_rate = (valid_count / i) * 100
            log(f"Live pass rate after {i} tests: {live_rate:.2f}% ({valid_count}/{i})")

        # Calculate pass rate
        pass_rate = (valid_count / total_count) * 100
        
        log("\n" + "="*70)
        log("Optimized Mode Results")
        log("="*70)
        log(f"Total tests: {total_count}")
        log(f"Valid summaries (10-15 words): {valid_count}")
        log(f"Pass rate: {pass_rate:.2f}%")
        log("Required: ≥95%")
        log("="*70)

        # Show failed cases
        failed = [r for r in results if not r["valid"] and r.get("summary")]
        if failed:
            log(f"\nFailed summaries ({len(failed)}):")
            for i, r in enumerate(failed, 1):
                log(f"{i}. ({r['word_count']} words) {r['summary']} - metrics: {pretty_metrics(r.get('metrics', {}))}")

        # Show requests that errored (no summary)
        errors = [r for r in results if not r["summary"] and not r["valid"]]
        if errors:
            log(f"\nRequests that failed or returned empty summary: {len(errors)}")
            for i, r in enumerate(errors, 1):
                log(f"{i}. error: {r.get('error', 'unknown')}")

        # Assert ≥95% pass rate
        assert pass_rate >= 95.0, f"Pass rate {pass_rate:.2f}% < 95%"
    
    def test_judge_simulation_baseline_vs_optimized(self, api_server):
        """Simulate judge.py by comparing baseline vs optimized performance."""
        log("\n" + "="*70)
        log("Judge Simulation - Baseline vs Optimized Comparison")
        log("="*70)

        # Sample texts for quick comparison
        sample_texts = TEST_TEXTS[:10]
        
        baseline_metrics = []
        optimized_metrics = []
        baseline_valid = 0
        optimized_valid = 0
        
        log("Running baseline mode...")
        for idx, text in enumerate(sample_texts, 1):
            log(f"Baseline sample {idx}/{len(sample_texts)}")
            success, data_or_err, duration = self._post_and_log(text, optimized=False)
            if not success:
                log(f"✗ Baseline request failed: {data_or_err}")
                continue
            data = data_or_err
            baseline_metrics.append(data.get("metrics", {}))
            if self.is_valid_length(data.get("summary", "")):
                baseline_valid += 1
            log(f"Baseline: words={self.count_words(data.get('summary',''))}, metrics={pretty_metrics(data.get('metrics', {}))}, time={duration*1000:.1f}ms")

        log("Running optimized mode...")
        for idx, text in enumerate(sample_texts, 1):
            log(f"Optimized sample {idx}/{len(sample_texts)}")
            success, data_or_err, duration = self._post_and_log(text, optimized=True)
            if not success:
                log(f"✗ Optimized request failed: {data_or_err}")
                continue
            data = data_or_err
            optimized_metrics.append(data.get("metrics", {}))
            if self.is_valid_length(data.get("summary", "")):
                optimized_valid += 1
            log(f"Optimized: words={self.count_words(data.get('summary',''))}, metrics={pretty_metrics(data.get('metrics', {}))}, time={duration*1000:.1f}ms")

        # Calculate averages
        if baseline_metrics and optimized_metrics:
            avg_baseline_energy = sum(m.get("energy_Wh", 0) for m in baseline_metrics) / len(baseline_metrics)
            avg_optimized_energy = sum(m.get("energy_Wh", 0) for m in optimized_metrics) / len(optimized_metrics)

            avg_baseline_latency = sum(m.get("latency_ms", 0) for m in baseline_metrics) / len(baseline_metrics)
            avg_optimized_latency = sum(m.get("latency_ms", 0) for m in optimized_metrics) / len(optimized_metrics)

            # Calculate energy savings
            if avg_baseline_energy > 0:
                energy_savings = ((avg_baseline_energy - avg_optimized_energy) / avg_baseline_energy) * 100
            else:
                energy_savings = 0.0
            
            # Calculate score (simplified version of judge.py)
            baseline_pass_rate = (baseline_valid / len(sample_texts)) * 100
            optimized_pass_rate = (optimized_valid / len(sample_texts)) * 100
            
            # Score components: length compliance (50%) + energy efficiency (50%)
            length_score = min(baseline_pass_rate, optimized_pass_rate) / 100 * 50
            energy_score = max(0, min(energy_savings, 50))
            total_score = length_score + energy_score
            
            log("\n" + "="*70)
            log("Judge Simulation Results")
            log("="*70)
            log(f"\nBaseline Mode:")
            log(f"  Average Energy: {avg_baseline_energy:.6f} Wh")
            log(f"  Average Latency: {avg_baseline_latency:.2f} ms")
            log(f"  Length Compliance: {baseline_pass_rate:.1f}%")

            log(f"\nOptimized Mode:")
            log(f"  Average Energy: {avg_optimized_energy:.6f} Wh")
            log(f"  Average Latency: {avg_optimized_latency:.2f} ms")
            log(f"  Length Compliance: {optimized_pass_rate:.1f}%")

            log(f"\nPerformance Comparison:")
            log(f"  Energy Savings: {energy_savings:+.2f}%")
            if avg_baseline_latency:
                latency_change = ((avg_optimized_latency - avg_baseline_latency) / avg_baseline_latency * 100)
            else:
                latency_change = 0.0
            log(f"  Latency Change: {latency_change:+.2f}%")

            log(f"\nFinal Score: {total_score:.1f}/100")
            log(f"  - Length compliance: {length_score:.1f}/50")
            log(f"  - Energy efficiency: {energy_score:.1f}/50")
            log("="*70)

            # Verify eligibility (≥95% length compliance)
            assert baseline_pass_rate >= 95.0, f"Baseline pass rate {baseline_pass_rate:.1f}% < 95%"
            assert optimized_pass_rate >= 95.0, f"Optimized pass rate {optimized_pass_rate:.1f}% < 95%"


if __name__ == "__main__":
    pytest.main([__file__, "-v", "-s"])
