"""Unit tests that can run without model download."""
import pytest
from unittest.mock import Mock, patch, MagicMock
from src.summarizer import count_words, normalize_text


class TestBasicFunctions:
    """Test basic utility functions."""
    
    def test_count_words(self):
        """Test word counting."""
        assert count_words("Un deux trois") == 3
        assert count_words("Test avec dix mots ici maintenant pour vérifier compte") == 9
        assert count_words("") == 0
    
    def test_normalize_text(self):
        """Test text normalization."""
        assert normalize_text("  extra   spaces  ") == "extra spaces"
        assert normalize_text("Texte\n\navec\tligne") == "Texte avec ligne"


class TestAPIEndpoint:
    """Test API endpoint logic (mocked)."""
    
    @patch('app.get_model')
    @patch('app.summarize')
    @patch('app.measure')
    def test_summarize_endpoint_baseline(self, mock_measure, mock_summarize, mock_get_model):
        """Test /summarize endpoint in baseline mode."""
        from app import app
        
        # Mock the model
        mock_model = Mock()
        mock_tokenizer = Mock()
        mock_get_model.return_value = (mock_model, mock_tokenizer)
        
        # Mock the summarize function
        mock_summarize.return_value = "Un résumé en français avec exactement douze mots pour tester."
        
        # Mock the measure function
        mock_measure.return_value = (
            "Un résumé en français avec exactement douze mots pour tester.",
            {
                "energy_Wh": 0.001234,
                "latency_ms": 123.45,
                "memory_MiB": 12.34
            }
        )
        
        client = app.test_client()
        
        response = client.post('/summarize', json={
            "text": "Test text here",
            "optimized": False
        })
        
        assert response.status_code == 200
        data = response.get_json()
        
        assert "summary" in data
        assert "metrics" in data
        assert "mode" in data
        assert data["mode"] == "baseline"
        assert 10 <= count_words(data["summary"]) <= 15
    
    @patch('app.get_model')
    @patch('app.summarize')
    @patch('app.measure')
    def test_summarize_endpoint_optimized(self, mock_measure, mock_summarize, mock_get_model):
        """Test /summarize endpoint in optimized mode."""
        from app import app
        
        # Mock the model
        mock_model = Mock()
        mock_tokenizer = Mock()
        mock_get_model.return_value = (mock_model, mock_tokenizer)
        
        # Mock the summarize function
        mock_summarize.return_value = "Résumé optimisé avec les bons paramètres et nombre de mots."
        
        # Mock the measure function
        mock_measure.return_value = (
            "Résumé optimisé avec les bons paramètres et nombre de mots.",
            {
                "energy_Wh": 0.000567,
                "latency_ms": 98.76,
                "memory_MiB": 8.90
            }
        )
        
        client = app.test_client()
        
        response = client.post('/summarize', json={
            "text": "Test text here",
            "optimized": True
        })
        
        assert response.status_code == 200
        data = response.get_json()
        
        assert data["mode"] == "optimized"
        assert 10 <= count_words(data["summary"]) <= 15
    
    def test_summarize_empty_text(self):
        """Test endpoint with empty text."""
        from app import app
        
        client = app.test_client()
        
        response = client.post('/summarize', json={
            "text": "",
            "optimized": False
        })
        
        assert response.status_code == 400
        data = response.get_json()
        assert "error" in data
    
    def test_summarize_text_too_long(self):
        """Test endpoint with text > 4000 characters."""
        from app import app
        
        client = app.test_client()
        
        response = client.post('/summarize', json={
            "text": "a" * 4001,
            "optimized": False
        })
        
        assert response.status_code == 400
        data = response.get_json()
        assert "error" in data
    
    def test_summarize_missing_text(self):
        """Test endpoint with missing text field."""
        from app import app
        
        client = app.test_client()
        
        response = client.post('/summarize', json={
            "optimized": False
        })
        
        assert response.status_code == 400
    
    def test_health_endpoint(self):
        """Test health check endpoint."""
        from app import app
        
        client = app.test_client()
        response = client.get('/health')
        
        assert response.status_code == 200
        data = response.get_json()
        assert data["status"] == "ok"


if __name__ == "__main__":
    pytest.main([__file__, "-v"])
