// Design4Green Web Application JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // Elements
    const inputText = document.getElementById('inputText');
    const charCount = document.getElementById('charCount');
    const summarizeBtn = document.getElementById('summarizeBtn');
    const outputSection = document.getElementById('outputSection');
    const summaryText = document.getElementById('summaryText');
    const wordCount = document.getElementById('wordCount');
    const energyValue = document.getElementById('energyValue');
    const latencyValue = document.getElementById('latencyValue');
    const memoryValue = document.getElementById('memoryValue');
    const modeDisplay = document.getElementById('modeDisplay');
    const loadingSpinner = document.getElementById('loadingSpinner');
    const errorMessage = document.getElementById('errorMessage');
    
    // Character counter
    inputText.addEventListener('input', function() {
        const count = inputText.value.length;
        charCount.textContent = count;
        
        if (count > 4000) {
            charCount.style.color = '#e74c3c';
        } else {
            charCount.style.color = '#7f8c8d';
        }
    });
    
    // Summarize button
    summarizeBtn.addEventListener('click', async function() {
        const text = inputText.value.trim();
        
        // Validate input
        if (!text) {
            showError('Veuillez entrer du texte à résumer.');
            return;
        }
        
        if (text.length > 4000) {
            showError('Le texte ne doit pas dépasser 4000 caractères.');
            return;
        }
        
        // Get selected mode
        const modeRadio = document.querySelector('input[name="mode"]:checked');
        const optimized = modeRadio.value === 'optimized';
        
        // Show loading state
        summarizeBtn.disabled = true;
        loadingSpinner.style.display = 'block';
        outputSection.style.display = 'none';
        errorMessage.style.display = 'none';
        
        try {
            // Call API
            const response = await fetch('/api/summarize', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    text: text,
                    optimized: optimized
                })
            });
            
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Erreur lors de la génération du résumé');
            }
            
            // Display results
            displayResults(data);
            
        } catch (error) {
            showError('Erreur: ' + error.message);
        } finally {
            // Hide loading state
            summarizeBtn.disabled = false;
            loadingSpinner.style.display = 'none';
        }
    });
    
    function displayResults(data) {
        // Display summary
        summaryText.textContent = data.summary;
        
        // Count words
        const words = data.summary.trim().split(/\s+/).length;
        wordCount.textContent = words;
        
        // Display metrics
        energyValue.textContent = formatNumber(data.metrics.energy_Wh, 6);
        latencyValue.textContent = formatNumber(data.metrics.latency_ms, 2);
        memoryValue.textContent = formatNumber(data.metrics.memory_MiB, 2);
        
        // Display mode
        modeDisplay.textContent = data.mode;
        
        // Show output section
        outputSection.style.display = 'block';
        
        // Scroll to results
        outputSection.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    function formatNumber(num, decimals) {
        if (typeof num !== 'number') {
            return '-';
        }
        return num.toFixed(decimals);
    }
    
    function showError(message) {
        errorMessage.textContent = message;
        errorMessage.style.display = 'block';
        
        // Scroll to error
        errorMessage.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    // Allow Enter key with Ctrl/Cmd to submit
    inputText.addEventListener('keydown', function(e) {
        if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
            summarizeBtn.click();
        }
    });
});